<?php

// Be careful if refactoring!
// Methods are called dynamically ($this->$method).
// Static code analyzers may report private methods as unused

require_once 'include/database/DateRange.php';

class WebhookVarEvaluator {

	static $TYPEMAP = [
		'varchar' =>  'string',
		'text' => 'string',
		'name' => 'string',
		'id' => 'string',
		'bool' => 'bool',
		'date' => 'datetime',
		'datetime' => 'datetime',
		'enum' => 'enum',
		'status' => 'enum',
		'multienum' => 'enum',
		'double' => 'number',
		'int' => 'number',
		'fax' => 'string',
		'phone' => 'string',
		'email' => 'string',
		'currency' => 'number',
		'base_currency' => 'number'
	];


	private $cache = [];
	var $rowUpdate;
	var $conditions;
	var $user_id;
	var $range;

	public function __construct($as_user, $rowUpdate, $conditions) {
		$this->rowUpdate = $rowUpdate;
		$this->conditions = $conditions;
		$this->user_id = $as_user;
		$this->range = new IAHDateRange;
	}

	function evaluate_when_changed($cond) {
		return array_key_exists($cond['field'], $this->rowUpdate->getRelatedData('webhookUpdate') ?? []);
	}

	function evaluate_when_not_changed($cond) {
		return !$this->evaluate_when_changed($cond);
	}

	function evaluate_when_current_value($cond) {
		return $this->evaluateValue($cond);
	}

	function evaluate_when_prev_value($cond) {
		return $this->evaluate_when_changed($cond) && $this->evaluateValue($cond, true);
	}

	function evaluate_when_changed_to($cond) {
		return $this->evaluate_when_changed($cond) && $this->evaluateValue($cond);
	}

	function evaluateValue($cond, $prev = false) {
		if ($prev) {
			$webhookOrig = $this->rowUpdate->getRelatedData('webhookOrig') ?? [];
			$value = array_get_default($webhookOrig, $cond['field']);
		} else {
			$value = $this->rowUpdate->getField($cond['field']);
		}
		$fdef = $this->rowUpdate->getFieldDefinition($cond['field']);
		if (!$fdef) return false;
		$type = array_get_default(self::$TYPEMAP, $fdef['type']);
		if (!$type) return false;
		$method = "evaluate_{$type}_{$cond['filter']}";
		if (!method_exists($this, $method)) $method = "evaluate_{$type}_catch_all";
		if (!method_exists($this, $method)) {
			return false;
		}
		return $this->$method($cond, $value);
	}

	private function date_within_interval($date, $interval) {
		return $date >= $interval[0] && $date <= $interval[1];
	}

	private function evaluate($cond) {
		if (isset($cond['glue'])) { // nested
			if (!isset($cond['conditions']) || !is_array($cond['conditions']))
				return null;
			$nested = new self($this->user_id, $this->rowUpdate, $cond['conditions']);
			$at_least_one = false;
			foreach (array_keys($cond['conditions']) as $idx) {
				$result = $nested($idx);
				if ($result === null) continue;
				$at_least_one = true;
				if ($result && $cond['glue'] != 'and') return true;
				if (!$result && $cond['glue'] == 'and') return false;
			}
			return ($cond['glue'] == 'and') && $at_least_one;
		}
		$field = array_get_default($cond, 'field');
		$when = array_get_default($cond, 'when');
		if (!$field || !$when) return null;
		$method = 'evaluate_when_' . $when;
		if (method_exists($this, $method))
			return $this->$method($cond);
		return null;
	}

	public function __invoke($index) {
		if (array_key_exists($index, $this->cache))
			return $this->cache[$index];
		$cond = array_get_default($this->conditions, $index);
		if (!$cond) {
			$this->cache[$index] = null;
			return null;
		}
		$result = $this->evaluate($cond);
		$this->cache[$index] = $result;
		return $result;
	}

	// string

	private function evaluate_string_eq($cond, $value) {
		return (string)$value === array_get_default($cond, 'value');
	}

	private function evaluate_string_not_eq($cond, $value) {
		return !$this->evaluate_string_eq($cond, $value);
	}

	private function evaluate_string_prefix($cond, $value) {
		echo $value, ' ', $cond['value'], "\n";
		return strpos((string)$value, array_get_default($cond, 'value')) === 0;
	}

	private function evaluate_string_suffix($cond, $value) {
		$value = (string)$value;
		$condValue = array_get_default($cond, 'value');
		$pos = strpos($value, $condValue);
		if ($pos === false) return false;
		return ($pos + strlen($condValue)) == strlen($value);
	}

	private function evaluate_string_like($cond, $value) {
		return strpos((string)$value, array_get_default($cond, 'value')) !== false;
	}

	private function evaluate_string_not_like($cond, $value) {
		return !$this->evaluate_string_like($cond, $value);
	}

	private function evaluate_string_empty($cond, $value) {
		return !strlen((string)$value);
	}

	private function evaluate_string_not_empty($cond, $value) {
		return !$this->evaluate_string_empty($cond, $value);
	}

	// number

	private function evaluate_number_lt($cond, $value) {
		return (double)array_get_default($cond, 'value') > (double)$value;
	}

	private function evaluate_number_gt($cond, $value) {
		return (double)array_get_default($cond, 'value') < (double)$value;
	}

	private function evaluate_number_lte($cond, $value) {
		return (double)array_get_default($cond, 'value') >= (double)$value;
	}

	private function evaluate_number_gte($cond, $value) {
		return (double)array_get_default($cond, 'value') <= (double)$value;
	}

	private function evaluate_number_between($cond, $value) {
		return (double)array_get_default($cond, 'value') <= (double)$value && (double)array_get_default($cond, 'value2') >= (double)$value;
	}

	private function evaluate_number_empty($cond, $value) {
		return !strlen($value);
	}

	private function evaluate_number_not_empty($cond, $value) {
		return !$this->evaluate_number_empty($cond, $value);
	}

	private function evaluate_number_eq($cond, $value) {
		return (double)array_get_default($cond, 'value') === (double)$value;
	}

	private function evaluate_number_not_eq($cond, $value) {
		return !$this->evaluate_number_eq($cond, $value);
	}

	// date / datetime

	private function evaluate_datetime_not_empty($cond, $value) {
		return !$this->evaluate_datetime_empty($cond, $value);
	}

	private function evaluate_datetime_empty($cond, $value) {
		return empty($value);
	}

	private function evaluate_datetime_before_date($cond, $value) {
		return $this->datetime_specific_date($cond, $value);
	}

	private function evaluate_datetime_after_date($cond, $value) {
		return $this->datetime_specific_date($cond, $value);
	}

	private function evaluate_datetime_on_date($cond, $value) {
		return $this->datetime_specific_date($cond, $value);
	}

	private function evaluate_datetime_not_on_date($cond, $value) {
		return $this->datetime_specific_date($cond, $value);
	}

	private function evaluate_datetime_yesterday($cond, $value) {
		return $this->datetime_relative_date($cond, $value);
	}

	private function evaluate_datetime_today($cond, $value) {
		return $this->datetime_relative_date($cond, $value);
	}

	private function evaluate_datetime_tomorrow($cond, $value) {
		return $this->datetime_relative_date($cond, $value);
	}

	private function evaluate_datetime_period_prev($cond, $value) {
		return $this->datetime_interval($cond, $value);
	}

	private function evaluate_datetime_period_current($cond, $value) {
		return $this->datetime_interval($cond, $value);
	}

	private function evaluate_datetime_period_next($cond, $value) {
		return $this->datetime_interval($cond, $value);
	}

	private function evaluate_bool_catch_all($cond, $value) {
		return (bool)((array_get_default($cond, 'filter') == 'is_false') ^ !!$value);
	}

	private function evaluate_enum_eq($cond, $value) {
		$value = $this->enum_is_multi($cond) ? array_filter(explode('^,^', $value), 'strlen') : [$value];
		$values = (array)array_get_default($cond, 'value');
		return count($values) == count($value) && count(array_diff($value, $values)) == 0;
	}

	private function evaluate_enum_not_eq($cond, $value) {
		return !$this->evaluate_enum_eq($cond, $value);
	}

	private function evaluate_enum_any_of($cond, $value) {
		$value = $this->enum_is_multi($cond) ? array_filter(explode('^,^', $value), 'strlen') : [$value];
		$values = (array)array_get_default($cond, 'value');
		$diff = array_diff($value, $values);
		return count($diff) < count($value);
	}

	private function evaluate_enum_not_any_of($cond, $value) {
		$value = $this->enum_is_multi($cond) ? array_filter(explode('^,^', $value), 'strlen') : [$value];
		$values = (array)array_get_default($cond, 'value');
		$diff = array_diff($value, $values);
		return count($diff) == count($value);
	}

	private function evaluate_enum_all_of($cond, $value) {
		$value = $this->enum_is_multi($cond) ? array_filter(explode('^,^', $value), 'strlen') : [$value];
		$values = (array)array_get_default($cond, 'value');
		$diff = array_diff($value, $values);
		return count($diff) == 0;
	}


	private function evaluate_enum_empty($cond, $value) {
		$value = $this->enum_is_multi($cond) ? array_filter(explode('^,^', $value), 'strlen') : [$value];
		return !!count($value);
	}

	private function evaluate_enum_not_empty($cond, $value) {
		return !$this->evaluate_enum_empty($cond, $value);
	}


	private function enum_is_multi($cond) {
		$fdef = $this->rowUpdate->getFieldDefinition($cond['field']);
		if (!$fdef) return false;
		return $fdef['type'] == 'multienum';
	}

	private function evaluate_datetime_between_dates($cond, $value) {
		global $timedate;
		$fdef = $this->rowUpdate->getFieldDefinition($cond['field']);
		if (!$fdef) return false;
		$start = trim(array_get_default($cond, 'value'));
		if (!preg_match('~^\d\d\d\d-\d\d-\d\d$~', $start)) return false;
		$end = trim(array_get_default($cond, 'value2'));
		if (!preg_match('~^\d\d\d\d-\d\d-\d\d$~', $end)) return false;
		if ($fdef['type'] == 'datetime') {
			$start = $timedate->handle_offset($start. ' 00:00:00', 'Y-m-d H:i:s', false, $this->user_id);
			$end = $timedate->handle_offset($end . ' 23:59:59', 'Y-m-d H:i:s', false, $this->user_id);
		}
		return $value >= $start && $value <= $end;
	}

	private function datetime_interval($cond, $value) {
		global $timedate;
		$fdef = $this->rowUpdate->getFieldDefinition($cond['field']);
		if (!$fdef) return false;
		switch ($cond['filter']) {
			case 'period_prev' : $mul = -1; break;
			case 'period_current': $mul = 0; break;
			case 'period_next': $mul = 1; break;
			default: return false;
		}
		switch ($cond['value']) {
			case 'day':
			case 'week':
			case 'month':
			case 'quarter':
			case 'year':
			case 'fiscal_quarter':
			case 'fiscal_year':
				list($start, $end) = $this->range->getPeriodStartEnd($mul, str_replace($cond['value'], '_', ''), 1);
				break;
			default:
				if (substr($cond['value'], 0, 5) != 'days_') { return false;}
				$number = (int)substr($cond['value'], 5);
				list($start, $end) = $this->range->getPeriodStartEnd($mul * $number, 'day', $number);
		}
		if ($fdef['type'] == 'date')
			$value = $value . ' 00:00:00';
		return $value >= $start && $value < $end;
	}

	private function datetime_relative_date($cond, $value) {
		switch ($cond['filter']) {
			case 'yesterday' : $offset = '-1 days'; break;
			case 'today' : $offset = ''; break;
			case 'tomorrow' : $offset = '+1 days'; break;
			default: return false;
		}
		global $timedate;
		$fdef = $this->rowUpdate->getFieldDefinition($cond['field']);
		if (!$fdef) return false;
		$format = $fdef['type'] == 'date' ? 'Y-m-d' : 'Y-m-d H:i:s';

		$now = gmdate('Y-m-d H:i:s', strtotime("now $offset"));
		$now = $timedate->handle_offset($now, 'Y-m-d', true, $this->user_id);
		$val = $timedate->handle_offset($value, 'Y-m-d', true, $this->user_id);
		return $val == $now;
	}

	private function datetime_specific_date($cond, $value) {
		global $timedate;
		$fdef = $this->rowUpdate->getFieldDefinition($cond['field']);
		if (!$fdef) return false;
		$date = trim(array_get_default($cond, 'value'));
		if (!preg_match('~^\d\d\d\d-\d\d-\d\d$~', $date)) return false;
		if ($fdef['type'] == 'datetime') {
			$start = $timedate->handle_offset($date . ' 00:00:00', 'Y-m-d H:i:s', false, $this->user_id);
			$end = $timedate->handle_offset($date . ' 23:59:59', 'Y-m-d H:i:s', false, $this->user_id);
		} else {
			$start = $end = $date;
		}
		switch ($cond['filter']) {
			case 'before' : return $value < $start;
			case 'after' : return $value > $end;
			case 'on_date' : return $value >= $start && $value <= $end;
			case 'not_on_date' : return $value < $start || $value > $end ;
			default: return false;
		}
	}

}


